//
// circles.c
// EZ Draw Program to Draw Nested Circles Based on a Fixed Specification.
// This is a recursion demonstration
//
// Thanks to Zane Helton for the original idea on his website:
//    https://dev.to/zanehelton/a-fractal-playground-with-processing-54gk
//
// CPSC 1070         Donald H. House         2/4/2019
// Clemson University
// 

#include <stdio.h>	// definitions for standard I/O routines
#include <math.h>	  // definitions for math routines

#include "ezdraw.h"

// Window Dimensions
#define WIDTH	        1000
#define HEIGHT		    800

// Smallest circle radius for the recursive base case
#define MINSIZE	      (HEIGHT / 20)

// Colors for screen background and clicked in points
#define BACKCOLOR     0, 0, 0         // black for background
#define POINTCOLOR    255, 50, 50     // red for points

// seven colors that will be cycled through depending on the recursion level
#define NUMCOLORS     7
unsigned char colors[7][3] = {{90, 173, 97}, {102, 102, 102}, {153, 112, 171}, {191, 129, 45}, {224, 130, 20}, {255, 127, 127}, {255, 80, 80}};

// handy boolean constants
typedef enum _bool{FALSE, TRUE} bool;

// struct storing the circle center (cx, cy) and its radius r
typedef struct _circle{
  int cx, cy, r;
} circle;

//
// Routine to recursively draw nested circles
//
void drawCircle(circle c, int whichcolor){
  circle rc;          // circle used for recursive calls
  int colorindex;
  
  // draw the circle with the current color
  EZ_SetColor(colors[whichcolor][0], colors[whichcolor][1], colors[whichcolor][2]);
  EZ_OutlineCircle(c.cx, c.cy, c.r);
  
  // recursively draw 4 circles inside if the circle radius is big enough
  if(c.r > MINSIZE){
    rc = c;                 // make recursive circle the same to begin with
    rc.r /= 2;              // but with half the radius

    // increase the color index modulo NUMCOLORS and draw the 4 new circles
    colorindex = (whichcolor + 1) % NUMCOLORS;
    
    // offset the recursive circle to each of the 4 quadrants
    // and recursively call drawCircle to draw each of the 4
    rc.cy += rc.r;
    drawCircle(rc, colorindex);
    rc.cy -= 2 * rc.r;
    drawCircle(rc, colorindex);
    rc.cy += rc.r;
    rc.cx += rc.r;
    drawCircle(rc, colorindex);
    rc.cx -= 2 * rc.r;
    drawCircle(rc, colorindex);
  }
}

//
// Display callback routine: clear the screen and draw the circle
// This is called every 1/30 of a second
//
void drawDisplay(){
  // the large outside circle
  circle mycircle = {WIDTH / 2, HEIGHT / 2, 2 * HEIGHT / 5};

  // clear the drawing to the background color
  EZ_SetBackColor(BACKCOLOR);
  EZ_ClearDrawing();
  
  // draw the circle using a recursive algorithm
  drawCircle(mycircle, 0);

  // display the drawing
  EZ_DisplayDrawing();
}

//
//  Main program to draw the nested circle. Quit on ESC or window kill
//
int main(){
  bool quit = FALSE;
  
  // start up EZ Draw
  EZ_Init(WIDTH, HEIGHT, "Nested Circle Demonstration");
  
  while(!quit){
    quit = EZ_HandleEvents(drawDisplay, NULL, NULL, NULL);
  }
  
  EZ_Quit();
  return 0;
}
