/*
 stri.c
 EZ Draw program to draw a set of Sierpinsky triangles, using mouse
 clicks to place the vertices. This is a recursion demonstration.
 
 CPSC 1070         Donald H. House         2/2/2019
 Clemson University
*/

#include <stdio.h>	// definitions for standard I/O routines
#include <math.h>	  // definitions for math routines

#include <ezdraw.h>

// Window Dimensions
#define WIDTH	        1000
#define HEIGHT		    800

// when triangle goes below this size, terminate recursion
#define MINSIZE	      (HEIGHT / 50)

// background, triangle, mouse point colors
#define BACKCOLOR     0, 0, 0
#define TRICOLOR      120, 120, 255
#define POINTCOLOR    255, 70, 70

// maximum number of triangles that will be drawn
#define MAXTRIS       100

// handy boolean constants
enum bool{FALSE, TRUE};

//
// Global variables to keep track of triangles and mouse clicks
//
EZ_Point triangle[3];         // array holding current triangle
EZ_Point clicks[3];           // points currently clicked in
enum bool havetriangle = FALSE;
int nclicks = 0;              // num mouse clicks since last triangle completed

// average the position of two points, giving the midpoint on the
// line segment between them
EZ_Point MidPoint(EZ_Point p0, EZ_Point p1){
  EZ_Point mid;
  
  mid.x = (int)((p0.x + p1.x) / 2.0 + 0.5);
  mid.y = (int)((p0.y + p1.y) / 2.0 + 0.5);
  
  return mid;
}

// A way to estimate the size of the triangle
// Find the smallest edge of a bounding rectangle
int size(EZ_Point p0, EZ_Point p1, EZ_Point p2){
  int minx, maxx, miny, maxy;
  int minsize;
  
  minx = maxx = p0.x;
  miny = maxy = p0.y;

  if(p1.x < minx)
    minx = p1.x;
  if(p1.x > maxx)
    maxx = p1.x;
  if(p1.y < miny)
    miny = p1.y;
  if(p1.y > maxy)
    maxy = p1.y;
  if(p2.x < minx)
    minx = p2.x;
  if(p2.x > maxx)
    maxx = p2.x;
  if(p2.y < miny)
    miny = p2.y;
  if(p2.y > maxy)
    maxy = p2.y;
  
  minsize = maxx - minx;
  if(maxy - miny < minsize)
    minsize = maxy - miny;

  return minsize;
}

//
// Routine to recursively draw a Sierpinski triangle
//
void drawSierpTriangle(EZ_Point p0, EZ_Point p1, EZ_Point p2){
  EZ_Point mid01, mid12, mid20;
  
  if(size(p0, p1, p2) < MINSIZE)    // if triangle is small, just draw a triangle
    EZ_OutlineTriangle(p0, p1, p2);
  else{                             // bigger triangle is drawn as 3 recursive triangles
    mid01 = MidPoint(p0, p1);
    mid12 = MidPoint(p1, p2);
    mid20 = MidPoint(p2, p0);
    drawSierpTriangle(p0, mid01, mid20);
    drawSierpTriangle(mid01, p1, mid12);
    drawSierpTriangle(mid12, p2, mid20);
  }
}

//
// Display callback routine: clear the screen and draw the triangles
// This is called every 1/30 of a second
//
void drawDisplay(){
  int i;
  
  // clear the drawing to the background color
  EZ_SetBackColor(BACKCOLOR);
  EZ_ClearDrawing();
  
  // draw the Sierpinski triangle if one has been clicked in
  if(havetriangle){
    EZ_SetColor(TRICOLOR);
    drawSierpTriangle(triangle[0], triangle[1], triangle[2]);
  }
  
  // draw all of the points that have been clicked in since the last triangle
  EZ_SetColor(POINTCOLOR);
  for(i = 0; i < nclicks; i++)
    EZ_FillCircle(clicks[i].x, clicks[i].y, 4);

  // display the drawing of all of the triangles
  EZ_DisplayDrawing();
}

//
// mouse button callback routine: record each click when the mouse button
// is released. Once 3 points have been clicked in, a triangle is complete
//
void handleMouseButton(int updown, int mousex, int mousey){
  int i;
  
  if(!updown){                //
    clicks[nclicks].x = mousex;
    clicks[nclicks].y = mousey;
    if(++nclicks == 3){
      nclicks = 0;
      havetriangle = TRUE;
      for(i = 0; i < 3; i++)
        triangle[i] = clicks[i];
    }
  }
}

//
// Main program to draw Sierpinski triangles. Quit on ESC or window kill
//
int main(){
  enum bool quit = FALSE;
  
  // start up the EZ Draw system
  EZ_Init(WIDTH, HEIGHT, "Sierpinski Triangle Demonstration");
  
  while(!quit){
    quit = EZ_HandleEvents(drawDisplay, NULL, handleMouseButton, NULL);
  }
  
  EZ_Quit();
  return 0;
}
