//
// ezdraw.c
// Source file for the EZ Draw system for use in introductory programming classes
//
// EZ Draw is built on top of SDL2, providing a highly simplified interface to
// many of the interactive graphics facilities of SDL. In particular, EZ Draw
// allows for keyboard interaction, left mouse button interaction, and mouse
// motion. It operates on a timer, causing a drawing event every 1/30 of a second.
// Drawing primitives include points, lines, and filled and open rectangles,
// circles, and triangles.
//
// This is version 1.0 of EZ Draw. It supports images and textures
//
// Donald H. House
// Oct. 3, 2019
// Clemson University
//
#include "ezdraw.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>

// defines used in circle drawing
#define PI        3.1415926536
#define RAD2DEG   (180.0 / PI)
#define DEG2RAD   (PI / 180.0)

#define EZ_TEXTURE_START_SIZE 10

// define boolean constants
//typedef enum _bool{FALSE, TRUE} bool;

// floating point (x,y) coordinates used for fill routines
// needing fractional pixel accuracy
typedef struct vector2f{
  float x, y;
} Vector2f;

// edge data structure used for scanline fill operations
typedef struct edge{
  int x0;             // starting x coordinate of edge (at its minimum y coordinate)
  int ymax;           // maximum y coordinate at end of edge
  float dxdy;         // inverse slope of line -- change in x for unit change in y
}Edge;

typedef struct _texture{
  SDL_Texture *tex;
  int w, h;
  bool dead;
} Texture;

//
// Global variables needed to record the state of the EZ Draw system
//
SDL_Window *EZ_Window = NULL;     // SDL window being used by EZ Draw
SDL_Renderer *EZ_Renderer = NULL; // SDL renderer being used by EZ Draw

SDL_Color EZ_Backcolor;     // current color for clearing the drawing
SDL_Color EZ_Drawcolor;     // current drawing color

int EZ_ScreenHeight;        // drawing window dimensions
int EZ_ScreenWidth;

int EZ_NumTextures;         // total number of currently loaded textures
int EZ_TextureArraySize;    // current size of the array of textures
Texture *EZ_Textures;       // array of textures, size of array is adjusted as needed

int shift = false;          // true if shift key depressed
int mousex, mousey;         // current mouse position if left button depressed

// routine to flip the y coordinate so for any calls to SDL routines
// in EZ Draw, y is positive upward. Note: in SDL it is positive downward
int Y(int y){
  return EZ_ScreenHeight - y;
}

// Assign the drawing color. Always opaque
void EZ_SetColor(unsigned char r, unsigned char g, unsigned char b){
  EZ_Drawcolor.r = r;
  EZ_Drawcolor.g = g;
  EZ_Drawcolor.b = b;
  EZ_Drawcolor.a = 255;
  SDL_SetRenderDrawColor(EZ_Renderer, EZ_Drawcolor.r, EZ_Drawcolor.g, EZ_Drawcolor.b, EZ_Drawcolor.a);
}

// Assign the drawing color. Can have alpha value
void EZ_SetColorRGBA(unsigned char r, unsigned char g, unsigned char b, unsigned char a){
  EZ_Drawcolor.r = r;
  EZ_Drawcolor.g = g;
  EZ_Drawcolor.b = b;
  EZ_Drawcolor.a = a;
  SDL_SetRenderDrawColor(EZ_Renderer, EZ_Drawcolor.r, EZ_Drawcolor.g, EZ_Drawcolor.b, EZ_Drawcolor.a);
}

// Assign the background color for clearning the drawing and images. Always opaque
void EZ_SetBackColor(unsigned char r, unsigned char g, unsigned char b){
  EZ_Backcolor.r = r;
  EZ_Backcolor.g = g;
  EZ_Backcolor.b = b;
  EZ_Backcolor.a = 255;
}

// Assign the background color for clearing the drawing and images. Can have alpha value
void EZ_SetBackColorRGBA(unsigned char r, unsigned char g, unsigned char b, unsigned char a){
  EZ_Backcolor.r = r;
  EZ_Backcolor.g = g;
  EZ_Backcolor.b = b;
  EZ_Backcolor.a = a;
}

// Draw a single point at (x, y)
void EZ_DrawPoint(int x, int y){
  SDL_RenderDrawPoint(EZ_Renderer, x, Y(y));
}

// Draw a single line from (x0, y0) to (x1, y1)
void EZ_DrawLine(int x0, int y0, int x1, int y1){
  SDL_RenderDrawLine(EZ_Renderer, x0, Y(y0), x1, Y(y1));
}

// Draw connected set of lines between the npoints in the points array
void EZ_DrawLineStrip(EZ_Point *points, int npoints){
  int i;
  for(i = 1; i < npoints; i++)
    EZ_DrawLine(points[i - 1].x, points[i - 1].y, points[i].x, points[i].y);
}

// Draw connected and closed set of lines between the npoints in the points array
void EZ_DrawLineLoop(EZ_Point *points, int npoints){
  int i;
  for(i = 1; i < npoints; i++)
    EZ_DrawLine(points[i - 1].x, points[i - 1].y, points[i].x, points[i].y);
  EZ_DrawLine(points[npoints - 1].x, points[npoints - 1].y, points[0].x, points[0].y);
}

// Draw an outlined rectangle. (xll, yll) lower left corner, w: width, h: height
void EZ_OutlineRect(int xll, int yll, int w, int h){
  SDL_Rect rect;
  rect.x = xll; rect.y = Y(yll + h);
  rect.w = w; rect.h = h;
  SDL_RenderDrawRect(EZ_Renderer, &rect);
}

// Draw nrects outlined rectangles using the rectangles in the rects array
void EZ_OutlineRects(EZ_Rect *rects, int nrects){
  int i;
  for(i = 0; i < nrects; i++)
    EZ_OutlineRect(rects[i].x, rects[i].y, rects[i].w, rects[i].h);
}

// Draw a filled in rectangle. (xll, yll) lower left corner, w: width, h: height
void EZ_FillRect(int xll, int yll, int w, int h){
  SDL_Rect rect;
  rect.x = xll; rect.y = Y(yll + h);
  rect.w = w; rect.h = h;
  SDL_RenderFillRect(EZ_Renderer, &rect);
}

// Draw nrects filled in rectangles using the rectangles in the rects array
void EZ_FillRects(EZ_Rect *rects, int nrects){
  int i;
  for(i = 0; i < nrects; i++)
    EZ_FillRect(rects[i].x, rects[i].y, rects[i].w, rects[i].h);
}

// draw a filled in triangle whose 3 vertices are specified by p0, p1, p2
// this is a scanline algorithm that goes from the bottom of the triangle
// to the top, drawing lines between the 2 edges intersecting the scanline
void EZ_FillTriangle(EZ_Point p0, EZ_Point p1, EZ_Point p2){
  
  Vector2f vtx[3], vtemp;   // floating point vertices
  int i, j;

  Edge etable[3];   // table of all edges
  Edge active[2];   // table of edges currently active
  int y;            // y coordinate of current scanline
  float xl, xr;     // left and right x coordinates at current scanline

  // insertion sort the vertices in ascending order by y
  // Also, make the coordinates floating point for line calculation later
  vtx[0].x = p0.x; vtx[0].y = Y(p0.y);
  vtx[1].x = p1.x; vtx[1].y = Y(p1.y);
  vtx[2].x = p2.x; vtx[2].y = Y(p2.y);
  for(i = 1; i < 3; i++){
    vtemp = vtx[i];
    for(j = i; j > 0 && vtemp.y < vtx[j - 1].y; j--)
      vtx[j] = vtx[j - 1];
    vtx[j] = vtemp;
  }
  
  // construct edge table in ascending order by y of lowest vertex
  // edge 0-1
  etable[0].x0 = vtx[0].x;      // starting x coord of edge
  etable[0].ymax = vtx[1].y;    // maximum y coord of edge
  etable[0].dxdy = (vtx[1].x - vtx[0].x) / (vtx[1].y - vtx[0].y); // edge slope inverse
  // edge 0-2
  etable[1].x0 = vtx[0].x;
  etable[1].ymax = vtx[2].y;
  etable[1].dxdy = (vtx[2].x - vtx[0].x) / (vtx[2].y - vtx[0].y);
  // edge 1-2
  etable[2].x0 = vtx[1].x;
  etable[2].ymax = vtx[2].y;
  etable[2].dxdy = (vtx[2].x - vtx[1].x) / (vtx[2].y - vtx[1].y);
  
  // place first 2 candidate edges in active table in descending
  // order by slope (note: smaller dxdy is larger slope)
  if(etable[0].dxdy < etable[1].dxdy){
    active[0] = etable[0];
    active[1] = etable[1];
  }
  else{
    active[0] = etable[1];
    active[1] = etable[0];
  }
  
  // if the first edge is horizontal, skip it
  if(active[0].dxdy == -INFINITY){
    active[0] = active[1];
    active[1] = etable[2];
  }
  else if(active[0].dxdy == INFINITY){
    active[0] = etable[2];
  }
  
  // both edges start at the x coordinate of the lowest vertex
  xl = active[0].x0;
  xr = active[1].x0;
  // loop from the bottom of the triangle to the top
  for(y = vtx[0].y; y <= vtx[2].y; y++){
    if(y == active[0].ymax)
      active[0] = etable[2];
    else if(y == active[1].ymax)
      active[1] = etable[2];
    
    SDL_RenderDrawLine(EZ_Renderer, xl, y, xr, y);

    xl += active[0].dxdy;   // increment x coordinates for next line
    xr += active[1].dxdy;
   }
}

// draw an outlined triangle whose 3 vertices are specified by p0, p1, p2
void EZ_OutlineTriangle(EZ_Point p0, EZ_Point p1, EZ_Point p2){
  SDL_Point vtx[4];
  vtx[0].x = vtx[3].x = p0.x; vtx[0].y = vtx[3].y = Y(p0.y);
  vtx[1].x = p1.x; vtx[1].y = Y(p1.y);
  vtx[2].x = p2.x; vtx[2].y = Y(p2.y);

  SDL_RenderDrawLines(EZ_Renderer, vtx, 4);
}

// draw a filled in circle whose center is (cx, cy) and has specified radius
void EZ_FillCircle(int cx, int cy, int radius) {
  float xoffset;
  int dx, dy;

  // scanline fill of circle
  // loop from bottom to top of circle, drawing lines from left to right side
  for (dy = -radius; dy <= radius; dy++) {
    xoffset = sqrt((radius * radius) - (dy * dy)); // horizontal offset
    dx = (int)xoffset;
  
    SDL_RenderDrawLine(EZ_Renderer, cx - dx, Y(cy + dy), cx + dx, Y(cy + dy));
  }
}

// draw an outlined in circle whose center is (cx, cy) and has specified radius
void EZ_OutlineCircle(int cx, int cy, int radius) {
  const int dangle = 2;
  const int npoints = 360 / dangle;
  SDL_Point points[npoints + 1];    // 1 extra point to close the circle
  int n;
  float angle;
  
  // draw a straight line for every 2 degrees around the circle
  for(n = 0; n < npoints; n++){
    angle = n * dangle * (DEG2RAD);
    points[n].x = radius * cos(angle) + cx;
    points[n].y = Y(radius * sin(angle) + cy);
  }
  points[npoints] = points[0];    // loop back to start to close circle
  
  SDL_RenderDrawLines(EZ_Renderer, points, npoints + 1);
}

// Create a blank image, filled with the current background color
// returns a pointer to the new image
EZ_Image *EZ_CreateBlankImage(int width, int height){
  EZ_Image* newimage;
  int rmask, gmask, bmask, amask;
  
 #if SDL_BYTEORDER == SDL_BIG_ENDIAN
      rmask = 0xff000000;
      gmask = 0x00ff0000;
      bmask = 0x0000ff00;
      amask = 0x000000ff;
  #else
      rmask = 0x000000ff;
      gmask = 0x0000ff00;
      bmask = 0x00ff0000;
      amask = 0xff000000;
  #endif

  newimage = SDL_CreateRGBSurface(0, width, height, 32, rmask, gmask, bmask, amask);
  if(newimage == NULL){
    SDL_DestroyRenderer(EZ_Renderer);
    SDL_DestroyWindow(EZ_Window);
    fprintf(stderr, "EZ Draw: error creating blank image of size (%d, %d): %s\n", width, height, SDL_GetError());
    SDL_Quit();
    exit(1);
  }
  
  Uint32 *ptr = (Uint32 *)newimage->pixels;
  #if SDL_BYTEORDER == SDL_BIG_ENDIAN
    Uint32 color = EZ_Backcolor.r << 24 | EZ_Backcolor.g << 16 | EZ_Backcolor.b << 8 | EZ_Backcolor.a;
  #else
    Uint32 color = EZ_Backcolor.a << 24 | EZ_Backcolor.b << 16 | EZ_Backcolor.g << 8 | EZ_Backcolor.r;
  #endif
  
  for(int row = 0; row < newimage->h; row++)
    for(int col = 0; col < newimage->w; col++)
      *ptr++ = color;

  return newimage;
}

// Create an image by loading from a bmp file,
// returns a pointer to the new image
EZ_Image *EZ_LoadBMPImage(const char *bmpfilename){
  EZ_Image *newimage, *image32;
  Uint32 *pixmap;
  unsigned char *ptr;
  int row, col;
  unsigned char r, g, b;
  
  newimage = SDL_LoadBMP(bmpfilename);
  if(newimage == NULL){
    SDL_DestroyRenderer(EZ_Renderer);
    SDL_DestroyWindow(EZ_Window);
    fprintf(stderr, "EZ Draw: error loading bmp file %s: %s\n", bmpfilename, SDL_GetError());
    SDL_Quit();
    exit(1);
  }
  
  if(newimage->format->BitsPerPixel != 32){
    switch(newimage->format->BitsPerPixel){
      case 24:
        image32 = EZ_CreateBlankImage(newimage->w, newimage->h);
        pixmap = (Uint32 *)image32->pixels;

        ptr = (unsigned char *)newimage->pixels;
        for(row = 0; row < newimage->h; row++){
          for(col = 0; col < newimage->w; col++){
            #if SDL_BYTEORDER == SDL_BIG_ENDIAN
              r = *ptr++;
              g = *ptr++;
              b = *ptr++;
              pixmap[row * newimage->w + col] = r << 24 | g << 16 | b << 8 | 255;
            #else
              b = *ptr++;
              g = *ptr++;
              r = *ptr++;
              pixmap[row * newimage->w + col] = 255 << 24 | b << 16 | g << 8 | r;
            #endif
          }
          ptr += newimage->pitch - newimage->w * 3;
        }
  
        SDL_FreeSurface(newimage);
        newimage = image32;
        break;
        
      case 32:
        break;
        
      default:
        SDL_FreeSurface(newimage);
        SDL_DestroyRenderer(EZ_Renderer);
        SDL_DestroyWindow(EZ_Window);
        fprintf(stderr, "EZ Draw: bmp file %s is in an unsupported format\n", bmpfilename);
        SDL_Quit();
        exit(1);
    }
  }
  return newimage;
}

// Create a texture from an image, and return its integer ID number.
// The texture will be loaded onto the graphics card, but will not be displayed
// until a call to EZ_DrawTexture() is made using its texture ID
int EZ_CreateTexture(EZ_Image *image){
  SDL_Texture* newtexture;

  if(EZ_NumTextures == EZ_TextureArraySize){
    EZ_Textures = (Texture *)realloc(EZ_Textures, EZ_TextureArraySize * 2 * sizeof(Texture));
    if(EZ_Textures == NULL){
      SDL_DestroyRenderer(EZ_Renderer);
      SDL_DestroyWindow(EZ_Window);
      fprintf(stderr, "EZ Draw: error allocating space for texture\n");
      SDL_Quit();
      exit(2);
    }
    EZ_TextureArraySize *= 2;
  }
  
  newtexture = SDL_CreateTextureFromSurface(EZ_Renderer, image);
  if(newtexture == NULL){
    SDL_DestroyRenderer(EZ_Renderer);
    SDL_DestroyWindow(EZ_Window);
    fprintf(stderr, "EZ Draw: error creating texture from image: %s\n", SDL_GetError());
    SDL_Quit();
    exit(3);
  }

  EZ_Textures[EZ_NumTextures].tex = newtexture;
  EZ_Textures[EZ_NumTextures].w = image->w;
  EZ_Textures[EZ_NumTextures].h = image->h;
  EZ_Textures[EZ_NumTextures].dead = false;

  return EZ_NumTextures++;
}

// Destroy a texture by removing it from the graphics card
// Its texture ID number will be made invalid after this call, and won't be reused
void EZ_DestroyTexture(int texture){
  if(texture < 0 || texture >= EZ_NumTextures || EZ_Textures[texture].dead){
    fprintf(stderr, "EZ Draw: trying to destroy invalid texture number %d\n", texture);
    exit(4);
  }

  SDL_DestroyTexture(EZ_Textures[texture].tex);
  EZ_Textures[texture].dead = true;
}

// Draw the texture indicated by the texture number
// texture_rect determines a rectangle on the texture image that will be drawn.
// drawing_rect determines a rectangle in the window that the texture image will be drawn to,
// if necessary, the portion of the image determined by texture_rect will be resized to
// fit drawing_rect
// If texture_rect is NULL, the entire texture will be drawn
// If drawing_rect is NULL, the texture will be drawn to the entire window
void EZ_DrawTexture(int texture, EZ_Rect *texture_rect, EZ_Rect *drawing_rect){
  EZ_Rect trect;
  EZ_Rect drect;

  if(texture < 0 || texture >= EZ_NumTextures || EZ_Textures[texture].dead){
    fprintf(stderr, "EZ Draw: trying to draw invalid texture number %d\n", texture);
    exit(4);
  }

  if(texture_rect == NULL){
    trect.x = 0;
    trect.y = 0;
    trect.w = EZ_Textures[texture].w;
    trect.h = EZ_Textures[texture].h;
  }
  else{
    trect = *texture_rect;
    trect.y = EZ_Textures[texture].h - (trect.y + trect.h);
  }
  
  if(drawing_rect == NULL){
    drect.x = 0;
    drect.y = 0;
    drect.w = EZ_ScreenWidth;
    drect.h = EZ_ScreenHeight;
  }
  else{
    drect = *drawing_rect;
    drect.y = Y(drect.y + drect.h);
  }
  
  SDL_RenderCopy(EZ_Renderer, EZ_Textures[texture].tex, &trect, &drect);
}

// This timer callback pushes an SDL_USEREVENT event into the queue,
// and causes itself to be called again at the same interval
Uint32 TimerCallBack(Uint32 interval, void *param)
{
  SDL_Event event;
  SDL_UserEvent userevent;
  
  if((int *)param == 0){ // avoids warning message for unused parameter
    ;
  }
  
  userevent.type = SDL_USEREVENT;
  userevent.code = 0;
  userevent.data1 = NULL;
  userevent.data2 = NULL;
  
  event.type = SDL_USEREVENT;
  event.user = userevent;
  
  SDL_PushEvent(&event);
  
  return interval;
}

// Initialize EZ Draw by starting up SDL, creating the drawing window,
// providing default colors, and setting up the 1/30 second timer
int EZ_Init(int width, int height, const char *title){
  Uint32 delay = (1000 / 30);   // establish timer delay of 1/30 second
  
  EZ_ScreenWidth = width;
  EZ_ScreenHeight = height;
  
  if(SDL_Init(SDL_INIT_VIDEO|SDL_INIT_TIMER) != 0){
    SDL_Log("EZ Draw: Unable to initialize ezdraw: %s", SDL_GetError());
    return 1;
  }
  
  if(title == NULL)     // avoid a NULL title, to make blank use ""
    title = "EZ Draw";
  
  EZ_Window = SDL_CreateWindow(title, SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED, width, height, 0);
  if(!EZ_Window){
    SDL_Log("EZ Draw: Unable to create window: %s", SDL_GetError());
    SDL_Quit();
    return 2;
  }

  EZ_Renderer = SDL_CreateRenderer(EZ_Window, -1, SDL_RENDERER_ACCELERATED);
  if(!EZ_Renderer){
    SDL_Log("EZ Draw: Unable to create renderer: %s", SDL_GetError());
    SDL_DestroyWindow(EZ_Window);
    SDL_Quit();
    return 3;
  }
  
  // default to drawing in black over a white background
  EZ_Drawcolor.r = EZ_Drawcolor.g = EZ_Drawcolor.b = 0; EZ_Drawcolor.a = 255;
  EZ_Backcolor.r = EZ_Backcolor.g = EZ_Backcolor.b = EZ_Backcolor.a = 255;
  
  // start up a timer that will be used to drive the display at 30 Hz
  SDL_AddTimer(delay, TimerCallBack, NULL);
  
  EZ_NumTextures = 0;
  EZ_TextureArraySize = EZ_TEXTURE_START_SIZE;
  EZ_Textures = (Texture *)malloc(EZ_TextureArraySize * sizeof(Texture));
  
  return 0;
}

// Clear the current drawing to the background color
void EZ_ClearDrawing(){
  SDL_Color savecolor = EZ_Drawcolor;
  
  SDL_SetRenderDrawColor(EZ_Renderer, EZ_Backcolor.r, EZ_Backcolor.g, EZ_Backcolor.b, EZ_Backcolor.a);
  SDL_RenderClear(EZ_Renderer);
  
  SDL_SetRenderDrawColor(EZ_Renderer, savecolor.r, savecolor.g, savecolor.b, savecolor.a);
}

// Display the current drawing in the display window
void EZ_DisplayDrawing(){
  SDL_RenderPresent(EZ_Renderer);
}

// self contained event loop that loops forever until ESC or window kill
void EZ_WaitForQuit(){
  SDL_Event event;
  const Uint8* keystate;
  
  while ( true ) {
    if (SDL_PollEvent(&event)) {
      if (event.type == SDL_QUIT) {
        break;
      }
      else if (event.type == SDL_KEYDOWN){
        keystate = SDL_GetKeyboardState(NULL);
        if (keystate[SDL_SCANCODE_ESCAPE])
          break;
      }
    }
  }
}

// Default drawing callback routine.
// if the caller to EZ_HandleEvents() does not specify a drawing callback
// this display is drawn: an X'd out circle
void errorDisplay(){
  EZ_SetBackColor(0, 0, 0);
  EZ_ClearDrawing();
  
  EZ_SetColor(255, 255, 255);
  EZ_FillCircle(EZ_ScreenWidth / 2, EZ_ScreenHeight / 2, EZ_ScreenHeight / 2);
  EZ_SetColor(255, 0, 0);
  EZ_DrawLine(0, 0, EZ_ScreenWidth, EZ_ScreenHeight);
  EZ_DrawLine(0, EZ_ScreenHeight, EZ_ScreenWidth, 0);
  
  EZ_DisplayDrawing();
}

// return the ASCII code corresponding to all of the single character keynames
// returned by SDL
unsigned char getKeyASCIICode(int shift, char keyname){
  // shiftcode contains ASCII character for the corresponding character in
  // keycode, when the shift key depressed
  unsigned char keycode[] = {'`', '-', '=', '[', ']', '\\', ';', '\'', ',', '.', '/'};
  unsigned char shiftcode[] = {'~', '_', '+', '{', '}', '|', ':', '\"', '<', '>', '?'};
  
  // shiftdigit contains the ASCII characters corresponding to shifted numeric keys
  unsigned char shiftdigit[] = {')', '!', '@', '#', '$', '%', '^', '&', '*', '('};
  
  unsigned char key;
  int i;
  
  key = keyname;    // by default use SDL keyname. It is upper case for alpha keys
  
  // if the shift key is depressed, need to find corresponding shifted character
  if(shift){
    if(isdigit(keyname))                // digits indexed directly from shiftdigit table
      key = shiftdigit[keyname - '0'];
    else if(!isalpha(keyname)){         // alphas do not need to be shifted
       // for non-alpha codes must search, since there is no natural ordering
      for(i = 0; i < 11 && keycode[i] != keyname; i++);
      key = shiftcode[i];
    }
  }else if(isalpha(keyname))  // alpha without shift needs to be converted to lower case
    key = tolower(keyname);
  
  return key;
}

// event dispatching routine, normally placed in a loop that loops until
// EZ_HandleEvents returns true (non zero), indicating an ESC key or window kill event
// The specified function is called if its event has occured.
int EZ_HandleEvents(void (*updateDisplay)(),
                    void (*handleKey)(unsigned char),
                    void (*handleButton)(int, int, int),
                    void (*handleMouseMotion)(int, int, int, int)){
  SDL_Event event;
  int dmousex, dmousey;
  Uint32 buttons;
  SDL_Keycode keycode;
  SDL_Scancode scancode;
  const char *keyname;
  unsigned char key;
  int done = false;           // we are not done unless ESC or window kill
  
  // if there is an SDL event, then see it is an EZ Draw event, and process it
  if(SDL_PollEvent(&event)){
    switch(event.type){
      case SDL_QUIT:                  // window kill event
        done = true;
        break;
      case SDL_USEREVENT:             // the 1/30 second timer event
        if(updateDisplay == NULL)     // if no drawing routine specified use default
          errorDisplay();
        else                          // otherwise call user's drawing routine
          updateDisplay();
        break;
      case SDL_KEYDOWN:               // keyboard key depressed event
        scancode = event.key.keysym.scancode;
        if(scancode == SDL_SCANCODE_ESCAPE)   // ESC key same as window kill
          done = true;
        else if(scancode == SDL_SCANCODE_LSHIFT || scancode == SDL_SCANCODE_RSHIFT)
          shift = true;               // if either shift key depressed, record it
        else if(handleKey != NULL){   // if the user has specified a keyboard callback
          keycode = SDL_GetKeyFromScancode(scancode);
          keyname = SDL_GetKeyName(keycode);
          // the single character key name strings are the printable keys
          // convert to ASCII, and send to user's keyboard callback routine
          if(strlen(keyname) == 1){
            key = getKeyASCIICode(shift, keyname[0]);
            handleKey(key);           // send ASCII key to user callback routine
          }
          else if(strcmp(keyname, "Space") == 0)  // space bar is only exception
            handleKey(' ');
        }
        break;
      case SDL_KEYUP:                   // keyboard key released event
        scancode = event.key.keysym.scancode;
        // if either shift key released, record it
        if(scancode == SDL_SCANCODE_LSHIFT || scancode == SDL_SCANCODE_RSHIFT)
          shift = false;
        break;
      case SDL_MOUSEBUTTONDOWN:         // mouse button down or up event
      case SDL_MOUSEBUTTONUP:
        // ignore if no mouse button callback or not left mouse button
        // get (x, y) coordinates and pass to user's mouse button callback routine
        if(handleButton != NULL && event.button.button == SDL_BUTTON_LEFT){
          mousex = event.button.x;        // save mouse coordinates for motion routine
          mousey = Y(event.button.y);
          handleButton((event.button.state == SDL_PRESSED), mousex, mousey);
        }
        break;
      case SDL_MOUSEMOTION:             // mouse motion event
        // ignore if no mouse motion callback
        if(handleMouseMotion != NULL){
          buttons = SDL_GetMouseState(NULL, NULL);
          // ignore if not left mouse button
          if(buttons & SDL_BUTTON(SDL_BUTTON_LEFT)){
            dmousex = event.motion.x - mousex;    // get change in mouse from previous
            dmousey = Y(event.motion.y) - mousey;
            // pass mouse position and change in position to motion callback routine
            handleMouseMotion(event.motion.x, Y(event.motion.y), dmousex, dmousey);
            // record mouse position for next time
            mousex = event.motion.x;
            mousey = Y(event.motion.y);
          }
        }
        break;
      default:
        break;
    }
  }
  
  return done;
}

// shut down EZ Draw by destroying SDL renderer, SDL window, and quitting SDL
void EZ_Quit(){
  SDL_DestroyRenderer(EZ_Renderer);
  SDL_DestroyWindow(EZ_Window);
  SDL_Quit();
}
