/******************************************************************************|
| DPA 8090 Example Code, Eric Patterson                                        |
| This is mostly plain C but uses a few things from C++ so needs C++ compiler. |
| Quaternion functions as well as OpenGL setup code here and                   |
| obj_parser, maths_func, gl_utils from Angton Gerdelan.                       |
| Accompanies written series "Anton's OpenGL 4 Tutorials"                      |
| http://antongerdelan.net/opengl/                                             |
| Email: anton at antongerdelan dot net                                        |
| Copyright Dr Anton Gerdelan, Trinity College Dublin, Ireland.                |
|******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <math.h>
#include <time.h>

#include <GL/glew.h>       // Include GLEW and new version of GL on Windows.
#include <GLFW/glfw3.h>    // GLFW helper library.

#include "maths_funcs.h"   // Anton's maths functions.
#include "gl_utils.h"      // Anton's opengl functions and small utilities like logs
#include "obj_parser.h"    // Anton's little Wavefront .obj mesh loader

#define _USE_MATH_DEFINES
#define ONE_DEG_IN_RAD (2.0 * M_PI) / 360.0 // 0.017444444

#define VERTEX_SHADER_FILE   "vs.glsl"
#define FRAGMENT_SHADER_FILE "fs.glsl"
#define MESH_FILE "teapot.obj"

/* create a unit quaternion q from an angle in degrees a, and an axis x,y,z */
void create_versor (float* q, float a, float x, float y, float z) {
	float rad = ONE_DEG_IN_RAD * a;
	q[0] = cosf (rad / 2.0f);
	q[1] = sinf (rad / 2.0f) * x;
	q[2] = sinf (rad / 2.0f) * y;
	q[3] = sinf (rad / 2.0f) * z;
}

/* convert a unit quaternion q to a 4x4 matrix m */
void quat_to_mat4 (float* m, float* q) {
	float w = q[0];
	float x = q[1];
	float y = q[2];
	float z = q[3];
	m[0] = 1.0f - 2.0f * y * y - 2.0f * z * z;
	m[1] = 2.0f * x * y + 2.0f * w * z;
	m[2] = 2.0f * x * z - 2.0f * w * y;
	m[3] = 0.0f;
	m[4] = 2.0f * x * y - 2.0f * w * z;
	m[5] = 1.0f - 2.0f * x * x - 2.0f * z * z;
	m[6] = 2.0f * y * z + 2.0f * w * x;
	m[7] = 0.0f;
	m[8] = 2.0f * x * z + 2.0f * w * y;
	m[9] = 2.0f * y * z - 2.0f * w * x;
	m[10] = 1.0f - 2.0f * x * x - 2.0f * y * y;
	m[11] = 0.0f;
	m[12] = 0.0f;
	m[13] = 0.0f;
	m[14] = 0.0f;
	m[15] = 1.0f;
}

/* normalise a quaternion in case it got a bit mangled */
void normalise_quat (float* q) {
	// norm(q) = q / magnitude (q)
	// magnitude (q) = sqrt (w*w + x*x...)
	// only compute sqrt if interior sum != 1.0
	float sum = q[0] * q[0] + q[1] * q[1] + q[2] * q[2] + q[3] * q[3];
	// NB: floats have min 6 digits of precision
	const float thresh = 0.0001f;
	if (fabs (1.0f - sum) < thresh) {
		return;
	}
	float mag = sqrt (sum);
	for (int i = 0; i < 4; i++) {
		q[i] = q[i] / mag;
	}
}

/* multiply quaternions to get another one. result=R*S */
void mult_quat_quat (float* result, float* r, float* s) {
	result[0] = s[0] * r[0] - s[1] * r[1] -
		s[2] * r[2] - s[3] * r[3];
	result[1] = s[0] * r[1] + s[1] * r[0] -
		s[2] * r[3] + s[3] * r[2];
	result[2] = s[0] * r[2] + s[1] * r[3] +
		s[2] * r[0] - s[3] * r[1];
	result[3] = s[0] * r[3] - s[1] * r[2] +
		s[2] * r[1] + s[3] * r[0];
	// re-normalise in case of mangling
	normalise_quat (result);
}

// The view and proj matrices make-up the camera position, orientation, fov, etc.
// Making them global simplifies things here.
// The position to start is purely arbitrary, and everything is relative, of course.
mat4 view_mat;
mat4 proj_mat;
vec3 cam_pos (0.03f, 0.0f, 5.0f);


int main () {
/*--------------------------------START OPENGL--------------------------------*/
	assert (restart_gl_log ());
	assert (start_gl ());        // start glfw window with GL context within

 
/*------------------------------CREATE GEOMETRY-------------------------------*/
	GLfloat* vp = NULL;    // array of vertex points
	GLfloat* vn = NULL;    // array of vertex normals
	GLfloat* vt = NULL;    // array of texture coordinates
	int point_count = 0;
	assert (load_obj_file (MESH_FILE, vp, vt, vn, point_count));

	// VAO -- vertex attribute objects bundle the various things associated with vertices
	GLuint vao;
	glGenVertexArrays (1, &vao);   // generating and binding is common pattern in OpenGL
	glBindVertexArray (vao);       // basically setting up memory and associating it

	// VBO -- vertex buffer object to contain coordinates
	GLuint points_vbo;
	glGenBuffers(1, &points_vbo);
	glBindBuffer(GL_ARRAY_BUFFER, points_vbo);
	glBufferData(GL_ARRAY_BUFFER, 3 * point_count * sizeof (GLfloat), vp, GL_STATIC_DRAW);
	glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, NULL);
	glEnableVertexAttribArray (0);

	// VBO -- normals -- needed for shading calcuations
	GLuint normals_vbo;
	glGenBuffers(1, &normals_vbo);
	glBindBuffer(GL_ARRAY_BUFFER, normals_vbo);
	glBufferData(GL_ARRAY_BUFFER, 3 * point_count * sizeof(GLfloat), vn, GL_STATIC_DRAW);
	glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, 0, NULL);
	glEnableVertexAttribArray (1);

	
/*-------------------------------CREATE SHADERS-------------------------------*/
        // The vertex shader program generally acts to transform vertices.
        // The fragment shader is where we'll do the actual "shading."

	GLuint shader_programme = create_programme_from_files (
		VERTEX_SHADER_FILE, FRAGMENT_SHADER_FILE
	);

        // model, view, and proj matrices act to transform verts from the model
        // description to the viewing space of the camera, to the final projection
	// a full "camera" description combines elements of the proj and view matrices
	int model_mat_location = glGetUniformLocation (shader_programme, "model_mat");
	int view_mat_location  = glGetUniformLocation (shader_programme, "view_mat");
	int proj_mat_location  = glGetUniformLocation (shader_programme, "proj_mat");
	
/*-------------------------------CREATE CAMERA--------------------------------*/
        // This sets up the matrix transformations that act as the camera lens
        // and sensor would to transform vertices from view space.
	float near = 0.1f;   // clipping planes
	float far = 1000.0f; 
	float fovy = 16.0f;  // vertical field of view, horiz calculated for aspect
	float aspect = (float)g_gl_width / (float)g_gl_height;      // aspect ratio
	proj_mat = perspective (fovy, aspect, near, far);
		
	float cam_speed = 5.0f; // 1 unit per second
	float cam_heading_speed = 100.0f; // 30 degrees per second
	float cam_heading = 0.0f; // y-rotation in degrees

	mat4 T = translate (
		identity_mat4 (), vec3 (-cam_pos.v[0], -cam_pos.v[1], -cam_pos.v[2])
	);

	mat4 R;               // 4x4 matrix with floats for rotation
	float quaternion[4];  // for negated initial camera orientation
	create_versor (quaternion, -cam_heading, 0.0f, 1.0f, 0.0f);
	quat_to_mat4 (R.m, quaternion); // convert quaternion to rotation matrix
	view_mat = R * T; // use the inverse orientation and transformation for view matrix
	vec4 fwd (0.0f, 0.0f, -1.0f, 0.0f); // F, R, and U are axes for camera orientation
	vec4 rgt (1.0f, 0.0f, 0.0f, 0.0f);  // F points forward, R to the right, and U up
	vec4 up (0.0f, 1.0f, 0.0f, 0.0f);   // They start world-axis aligned but can change.

	
/*---------------------------SET RENDERING DEFAULTS---------------------------*/
	// Choose vertex and fragment shaders to use as well as view and proj matrices.
	glUseProgram (shader_programme);
	glUniformMatrix4fv (view_mat_location, 1, GL_FALSE, view_mat.m);
	glUniformMatrix4fv (proj_mat_location, 1, GL_FALSE, proj_mat.m);

	// The model_mat stores the position and orientation transformations for our mesh.
	// Anton has setup some nice functions for scaling, rotating, and translating. 
	// These place the teapot in a good spot, scale it down some (vestigal), and orient it to start nicely.
	mat4 model_mat;   
	model_mat = translate (identity_mat4 () * scale(identity_mat4(), vec3(0.5, 0.5, 0.5)), vec3(0, -0.5, 0)) * rotate_y_deg (identity_mat4 (), 90 );

	// Setup some basic GL attributes.	
	glEnable (GL_DEPTH_TEST);   // enable depth-testing
	glDepthFunc (GL_LESS);      // depth-testing interprets a smaller value as "closer"
	glEnable (GL_CULL_FACE);    // cull face
	glCullFace (GL_BACK);       // cull back face
	glFrontFace (GL_CCW);       // set counter-clock-wise vertex order to mean the front
	glClearColor (0.1, 0.1, 0.1, 1.0);   // non-black background to help spot mistakes
	glViewport (0, 0, g_gl_width, g_gl_height); // make sure we keep our aspect ratio correct

	
/*-------------------------------RENDERING LOOP-------------------------------*/
	
	double curr_xpos, curr_ypos;
	double last_xpos, last_ypos;
	double diff_xpos, diff_ypos;
	curr_xpos = 0;
	curr_ypos = 0;
	last_xpos = 0;
	last_ypos = 0;
	diff_xpos = 0;
	diff_ypos = 0;

	while (!glfwWindowShouldClose (g_window)) {
		// update timers
		static double previous_seconds = glfwGetTime ();
		double current_seconds = glfwGetTime ();
		double elapsed_seconds = current_seconds - previous_seconds;
		previous_seconds = current_seconds;
		_update_fps_counter (g_window);
		
		// wipe the drawing surface clear
		glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
		// setup shader use	
		glUseProgram (shader_programme);

		// update and draw Tea Pots 
		model_mat = rotate_y_deg(identity_mat4(), -0.8) * model_mat;

		glUniformMatrix4fv (model_mat_location, 1, GL_FALSE, model_mat.m);
		glDrawArrays (GL_TRIANGLES, 0, point_count);

		// update other events like input handling 
		glfwPollEvents ();
		
		// control keys
		bool cam_moved = false;
		vec3 move (0.0, 0.0, 0.0);
		float cam_yaw = 0.0f; // y-rotation in degrees
		float cam_pitch = 0.0f;
		float cam_roll = 0.0;

		if (glfwGetKey (g_window, GLFW_KEY_A)) {
			move.v[0] -= cam_speed * elapsed_seconds;
			cam_moved = true;
		}
		if (glfwGetKey (g_window, GLFW_KEY_D)) {
			move.v[0] += cam_speed * elapsed_seconds;
			cam_moved = true;
		}
		if (glfwGetKey (g_window, GLFW_KEY_Q)) {
			move.v[1] += cam_speed * elapsed_seconds;
			cam_moved = true;
		}
		if (glfwGetKey (g_window, GLFW_KEY_E)) {
			move.v[1] -= cam_speed * elapsed_seconds;
			cam_moved = true;
		}
		if (glfwGetKey (g_window, GLFW_KEY_W)) {
			move.v[2] -= cam_speed * elapsed_seconds;
			cam_moved = true;
		}
		if (glfwGetKey (g_window, GLFW_KEY_S)) {
			move.v[2] += cam_speed * elapsed_seconds;
			cam_moved = true;
		}
		if (glfwGetKey (g_window, GLFW_KEY_LEFT)) {
			cam_yaw += cam_heading_speed * elapsed_seconds;
			cam_moved = true;
			
			// create a quaternion representing change in heading (the yaw)
			float q_yaw[4];
			create_versor (q_yaw, cam_yaw, up.v[0], up.v[1], up.v[2]);
			// add yaw rotation to the camera's current orientation
			mult_quat_quat (quaternion, q_yaw, quaternion);
			
			// recalc axes to suit new orientation
			quat_to_mat4 (R.m, quaternion);
			fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			rgt = R * vec4 (1.0, 0.0, 0.0, 0.0);
			up = R * vec4 (0.0, 1.0, 0.0, 0.0);
		}
		if (glfwGetKey (g_window, GLFW_KEY_RIGHT)) {
			cam_yaw -= cam_heading_speed * elapsed_seconds;
			cam_moved = true;
			float q_yaw[4];
			create_versor (q_yaw, cam_yaw, up.v[0], up.v[1], up.v[2]);
			mult_quat_quat (quaternion, q_yaw, quaternion);
			
			// recalc axes to suit new orientation
			quat_to_mat4 (R.m, quaternion);
			fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			rgt = R * vec4 (1.0, 0.0, 0.0, 0.0);
			up = R * vec4 (0.0, 1.0, 0.0, 0.0);
		}
		if (glfwGetKey (g_window, GLFW_KEY_UP)) {
			cam_pitch += cam_heading_speed * elapsed_seconds;
			cam_moved = true;
			float q_pitch[4];
			create_versor (q_pitch, cam_pitch, rgt.v[0], rgt.v[1], rgt.v[2]);
			mult_quat_quat (quaternion, q_pitch, quaternion);
			
			// recalc axes to suit new orientation
			quat_to_mat4 (R.m, quaternion);
			fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			rgt = R * vec4 (1.0, 0.0, 0.0, 0.0);
			up = R * vec4 (0.0, 1.0, 0.0, 0.0);
		}
		if (glfwGetKey (g_window, GLFW_KEY_DOWN)) {
			cam_pitch -= cam_heading_speed * elapsed_seconds;
			cam_moved = true;
			float q_pitch[4];
			create_versor (q_pitch, cam_pitch, rgt.v[0], rgt.v[1], rgt.v[2]);
			mult_quat_quat (quaternion, q_pitch, quaternion);
			
			// recalc axes to suit new orientation
			quat_to_mat4 (R.m, quaternion);
			fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			rgt = R * vec4 (1.0, 0.0, 0.0, 0.0);
			up = R * vec4 (0.0, 1.0, 0.0, 0.0);
		}
		if (glfwGetKey (g_window, GLFW_KEY_Z)) {
			cam_roll -= cam_heading_speed * elapsed_seconds;
			cam_moved = true;
			float q_roll[4];
			create_versor (q_roll, cam_roll, fwd.v[0], fwd.v[1], fwd.v[2]);
			mult_quat_quat (quaternion, q_roll, quaternion);
			
			// recalc axes to suit new orientation
			quat_to_mat4 (R.m, quaternion);
			fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			rgt = R * vec4 (1.0, 0.0, 0.0, 0.0);
			up = R * vec4 (0.0, 1.0, 0.0, 0.0);
		}
		if (glfwGetKey (g_window, GLFW_KEY_C)) {
			cam_roll += cam_heading_speed * elapsed_seconds;
			cam_moved = true;
			float q_roll[4];
			create_versor (q_roll, cam_roll, fwd.v[0], fwd.v[1], fwd.v[2]);
			mult_quat_quat (quaternion, q_roll, quaternion);
			
			// recalc axes to suit new orientation
			quat_to_mat4 (R.m, quaternion);
			fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			rgt = R * vec4 (1.0, 0.0, 0.0, 0.0);
			up = R * vec4 (0.0, 1.0, 0.0, 0.0);
		}
		float mouse_diff = 5.0;
		if (glfwGetMouseButton(g_window, GLFW_MOUSE_BUTTON_LEFT)){
			
			glfwGetCursorPos(g_window, &curr_xpos, &curr_ypos);

			diff_xpos = curr_xpos - last_xpos;
			diff_ypos = curr_ypos - last_ypos;
			last_xpos = curr_xpos;
			last_ypos = curr_ypos;

			if (diff_ypos > mouse_diff) {
			    mat4 R1 = rotate_x_deg (identity_mat4 (), -1.2);
			    vec4 temp = R1 * vec4 (cam_pos.v[0], cam_pos.v[1], cam_pos.v[2], 1.0);
			    cam_pos.v[0] = temp.v[0];
			    cam_pos.v[1] = temp.v[1];
			    cam_pos.v[2] = temp.v[2];

			    cam_moved = true;

			    cam_pitch -= 1.2;
			    float q_pitch[4];
			    create_versor (q_pitch, cam_pitch, rgt.v[0], rgt.v[1], rgt.v[2]);
			    mult_quat_quat (quaternion, q_pitch, quaternion);
			
			    // recalc axes to suit new orientation
			    quat_to_mat4 (R.m, quaternion);
			    fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			    rgt = R * vec4 (1.0, 0.0, 0.0, 0.0);
			}
			else if (diff_ypos < -mouse_diff) {
			    mat4 R1 = rotate_x_deg (identity_mat4 (), 1.2);
			    vec4 temp = R1 * vec4 (cam_pos.v[0], cam_pos.v[1], cam_pos.v[2], 1.0);
			    cam_pos.v[0] = temp.v[0];
			    cam_pos.v[1] = temp.v[1];
			    cam_pos.v[2] = temp.v[2];

			    cam_moved = true;

			    cam_pitch += 1.2;
			    float q_pitch[4];
			    create_versor (q_pitch, cam_pitch, rgt.v[0], rgt.v[1], rgt.v[2]);
			    mult_quat_quat (quaternion, q_pitch, quaternion);
			
			    // recalc axes to suit new orientation
			    quat_to_mat4 (R.m, quaternion);
			    fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			}

			if (diff_xpos > mouse_diff) {
			    mat4 R1 = rotate_y_deg (identity_mat4 (), -1.2);
			    vec4 temp = R1 * vec4 (cam_pos.v[0], cam_pos.v[1], cam_pos.v[2], 1.0);
			    cam_pos.v[0] = temp.v[0];
			    cam_pos.v[1] = temp.v[1];
			    cam_pos.v[2] = temp.v[2];

			    cam_moved = true;

			    cam_yaw -= 1.2;
			
			    // create a quaternion representing change in heading (the yaw)
			    float q_yaw[4];
			    create_versor (q_yaw, cam_yaw, up.v[0], up.v[1], up.v[2]);
			    // add yaw rotation to the camera's current orientation
			    mult_quat_quat (quaternion, q_yaw, quaternion);
			
			    // recalc axes to suit new orientation
			    quat_to_mat4 (R.m, quaternion);
			    fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			    rgt = R * vec4 (1.0, 0.0, 0.0, 0.0);
			}
			else if (diff_xpos < -mouse_diff) {
			    mat4 R1 = rotate_y_deg (identity_mat4 (), 1.2);
			    vec4 temp = R1 * vec4 (cam_pos.v[0], cam_pos.v[1], cam_pos.v[2], 1.0);
			    cam_pos.v[0] = temp.v[0];
			    cam_pos.v[1] = temp.v[1];
			    cam_pos.v[2] = temp.v[2];

			    cam_moved = true;

			    cam_yaw += 1.2;
			
			    // create a quaternion representing change in heading (the yaw)
			    float q_yaw[4];
			    create_versor (q_yaw, cam_yaw, up.v[0], up.v[1], up.v[2]);
			    // add yaw rotation to the camera's current orientation
			    mult_quat_quat (quaternion, q_yaw, quaternion);
			
			    // recalc axes to suit new orientation
			    quat_to_mat4 (R.m, quaternion);
			    fwd = R * vec4 (0.0, 0.0, -1.0, 0.0);
			}
		}

		if (glfwGetMouseButton(g_window, GLFW_MOUSE_BUTTON_MIDDLE)){
			
			glfwGetCursorPos(g_window, &curr_xpos, &curr_ypos);

			diff_xpos = curr_xpos - last_xpos;
			diff_ypos = curr_ypos - last_ypos;
			last_xpos = curr_xpos;
			last_ypos = curr_ypos;

			if (diff_ypos > mouse_diff) {
			    //cam_pos.v[1] += cam_speed * elapsed_seconds;
			    move.v[1] += cam_speed * elapsed_seconds;
			    cam_moved = true;
			}
			else if (diff_ypos < -mouse_diff) {
			    //cam_pos.v[1] -= cam_speed * elapsed_seconds;
			    move.v[1] -= cam_speed * elapsed_seconds;
			    cam_moved = true;
			}

			if (diff_xpos > mouse_diff) {
			    //cam_pos.v[0] -= cam_speed * elapsed_seconds;
			    move.v[0] -= cam_speed * elapsed_seconds;
			    cam_moved = true;
			}
			else if (diff_xpos < -mouse_diff) {
			    //cam_pos.v[0] += cam_speed * elapsed_seconds;
			    move.v[0] += cam_speed * elapsed_seconds;
			    cam_moved = true;
			}
		}

		if (glfwGetMouseButton(g_window, GLFW_MOUSE_BUTTON_RIGHT)){

			glfwGetCursorPos(g_window, &curr_xpos, &curr_ypos);

			diff_xpos = curr_xpos - last_xpos;
			diff_ypos = curr_ypos - last_ypos;
			last_xpos = curr_xpos;
			last_ypos = curr_ypos;

			if (diff_ypos > mouse_diff) {
			    move.v[2] -= cam_speed * elapsed_seconds;
			    cam_moved = true;
			}
			else if (diff_ypos < -mouse_diff) {
			    move.v[2] += cam_speed * elapsed_seconds;
			    cam_moved = true;
			}

			if (diff_xpos > mouse_diff) {
			    move.v[2] -= cam_speed * elapsed_seconds;
			    cam_moved = true;
			}
			else if (diff_xpos < -mouse_diff) {
			    move.v[2] += cam_speed * elapsed_seconds;
			    cam_moved = true;
			}

		}

			
		// update view matrix
		if (cam_moved) {
			quat_to_mat4 (R.m, quaternion);
			
			// checking for fp errors
			//	printf ("dot fwd . up %f\n", dot (fwd, up));
			//	printf ("dot rgt . up %f\n", dot (rgt, up));
			//	printf ("dot fwd . rgt\n %f", dot (fwd, rgt));

			cam_pos = cam_pos + vec3 (fwd) * -move.v[2];
			cam_pos = cam_pos + vec3 (up) * move.v[1];
			cam_pos = cam_pos + vec3 (rgt) * move.v[0];
			mat4 T = translate (identity_mat4 (), vec3 (cam_pos));
			
			view_mat = inverse (R) * inverse (T);
			glUniformMatrix4fv (view_mat_location, 1, GL_FALSE, view_mat.m);
		}
		
		
		if (GLFW_PRESS == glfwGetKey (g_window, GLFW_KEY_ESCAPE)) {
			glfwSetWindowShouldClose (g_window, 1);
		}

		
		aspect = (float)g_gl_width / (float)g_gl_height; // aspect ratio
		proj_mat = perspective (fovy, aspect, near, far);
	        glUniformMatrix4fv (proj_mat_location, 1, GL_FALSE, proj_mat.m);

		// put the stuff we've been drawing onto the display
		glfwSwapBuffers (g_window);
	}
	
	// close GL context and any other GLFW resources
	glfwTerminate();
	return 0;
}
