#version 410

// Demonstrate Phong or Blinn-Phong shading models.
// These are based on non-scientific "intensities."
// More akin to photometry rather than radiometry.
// See Blinn paper referenced on course website.
// Original code from Anton's examples.

// Inputs from application.
// Generally, "in" like the eye and normal vectors for things that change frequently,
// and "uniform" for things that change less often (think scene versus vertices).  
in vec3 position_eye, normal_eye;
uniform mat4 view_mat;

// This light setup would usually be passed in from the application.
vec3 light_position_world  = vec3 (10.0, 25.0, 10.0);
vec3 Ls = vec3 (1.0, 1.0, 1.0);    // neutral, full specular color of light
vec3 Ld = vec3 (0.8, 0.8, 0.8);    // neutral, lessened diffuse light color of light
vec3 La = vec3 (0.12, 0.12, 0.12); // ambient color of light - just a bit more than dk gray bg
  
// Surface reflectance properties for Phong or Blinn-Phong shading models below.
vec3 Ks = vec3 (1.0, 1.0, 1.0);    // fully reflect specular light
vec3 Kd = vec3 (0.32, 0.18, 0.5);  // purple diffuse surface reflectance
vec3 Ka = vec3 (1.0, 1.0, 1.0);    // fully reflect ambient light
float specular_exponent = 400.0;   // specular 'power' -- controls "roll-off"

// These come from the VAO for texture coordinates.
in vec2 texture_coords;

// And from the uniform outputs for the textures setup in main.cpp.
uniform sampler2D texture00;
uniform sampler2D texture01;

// Shader programs can also designate outputs -- in this case the final pixel color to draw.
out vec4 fragment_color;           // color of surface to draw in this case

void main () {
	// Ambient intensity
	vec3 Ia = La * Ka;

	// Kd.x = normal_eye.x;   // Uncomment these for fun -- shows the normals of the object.
	// Kd.y = normal_eye.y;
	// Kd.z = normal_eye.z;

        // These next few lines sample the current texture coord (s, t) in texture00 and 01 and mix.
        // Delete these for your procedural shader;  you'll only need the texture_coords for finding
        // where you are on your texture layout (UV Layout from Maya).  Use texture files only
        // if you make it to doing optional project extensions.
        vec4 texel_a = texture (texture00, texture_coords*4.0);
        vec4 texel_b = texture (texture01, texture_coords*4.0);
        vec4 mixed   = mix (texel_a, texel_b, texture_coords.x);
        Kd.x = mixed.x;
        Kd.y = mixed.y;
        Kd.z = mixed.z; 
  
	// Transform light position to view space.
	// Vectors here are appended with _eye as a reminder once in view space versus world space.
	// "Eye" is used instead of "camera" since reflectance models often phrased that way.
	vec3 light_position_eye = vec3 (view_mat * vec4 (light_position_world, 1.0));
	vec3 distance_to_light_eye = light_position_eye - position_eye;
	vec3 direction_to_light_eye = normalize (distance_to_light_eye);

	// Diffuse intensity
	float dot_prod = dot (direction_to_light_eye, normal_eye);
	dot_prod = max (dot_prod, 0.0);
	vec3 Id = Ld * Kd * dot_prod; // final diffuse intensity

	// Specular is view dependent; get vector toward camera.
	vec3 surface_to_viewer_eye = normalize (-position_eye);

        // Phong	
	//vec3 reflection_eye = reflect (-direction_to_light_eye, normal_eye);
	//float dot_prod_specular = dot (reflection_eye, surface_to_viewer_eye);
	//dot_prod_specular = max (dot_prod_specular, 0.0);
	//float specular_factor = pow (dot_prod_specular, specular_exponent);
	
	// Blinn
	vec3 half_way_eye = normalize (surface_to_viewer_eye + direction_to_light_eye);
	float dot_prod_specular = max (dot (half_way_eye, normal_eye), 0.0);
	float specular_factor = pow (dot_prod_specular, specular_exponent);
	
	// Specular intensity
	vec3 Is = Ls * Ks * specular_factor; // final specular intensity
	
	// final color
	fragment_color = vec4 (Is + Id + Ia, 1.0);
}
